<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Pendaftaran extends CI_Controller {

    public function __construct() {
        parent::__construct();
        if (!$this->session->userdata('id_user')) {
            redirect('auth/login');
        }
        $this->load->model(['Pendaftaran_model', 'User_model']);
        $this->load->library('form_validation');
        $this->load->helper(['form', 'url']);
    }
    
    // Perbaikan Index
    public function index() {
        $id_role = $this->session->userdata('id_role');
        if ($id_role == 2) {
            redirect('pendaftaran/form_perorangan');
        } elseif ($id_role == 3) {
            redirect('pendaftaran/form_kelompok');
        } else {
            // Arahkan ke dashboard admin atau halaman lain jika role tidak dikenal
            redirect('pendaftaran/dashboard'); 
        }
    }

    public function form_perorangan() {
    $data['jenis_seni'] = $this->Pendaftaran_model->get_all_jenis_seni();
    $this->load->view('pendaftaran/form_perorangan', $data);
}

// Metode submit_perorangan yang diperbaiki
public function submit_perorangan() {
    // Validasi yang lebih lengkap sesuai DB
    $this->form_validation->set_rules('nik', 'NIK', 'required|numeric|exact_length[16]|is_unique[sdm.nik]');
    $this->form_validation->set_rules('nama_lengkap', 'Nama Lengkap', 'required');
    $this->form_validation->set_rules('jenis_identitas', 'Jenis Identitas', 'required');
    $this->form_validation->set_rules('tempat_lahir', 'Tempat Lahir', 'required');
    $this->form_validation->set_rules('tanggal_lahir', 'Tanggal Lahir', 'required');
    $this->form_validation->set_rules('jenis_kelamin', 'Jenis Kelamin', 'required|in_list[L,P]');
    $this->form_validation->set_rules('agama', 'Agama', 'required');
    $this->form_validation->set_rules('provinsi', 'Provinsi', 'required');
    $this->form_validation->set_rules('kabupaten', 'Kabupaten/Kota', 'required');
    $this->form_validation->set_rules('no_hp', 'Nomor HP', 'required|numeric|min_length[7]|max_length[15]');

    // HAPUS VALIDASI BAWAAN UNTUK 'jenis_seni'
    // $this->form_validation->set_rules('jenis_seni', 'Jenis Seni', 'required'); 

    if ($this->form_validation->run() == FALSE) {
        $this->session->set_flashdata('error', validation_errors());
        $this->form_perorangan();
        return;
    }

    // LAKUKAN VALIDASI MANUAL UNTUK JENIS SENI
    $jenis_seni_array = $this->input->post('jenis_seni');
    if (empty($jenis_seni_array) || !is_array($jenis_seni_array)) {
        $this->session->set_flashdata('error', '<p>Jenis Seni wajib dipilih.</p>');
        $this->form_perorangan();
        return;
    }

    $this->db->trans_begin();
    try {
        // Upload files
        $foto_identitas = $this->upload_file('foto_identitas', 'foto_sdm');
        $pas_foto = $this->upload_file('pas_foto', 'foto_sdm');

        // Simpan data ke tabel 'sdm'
        $data_sdm = [
            'nik' => $this->input->post('nik'),
            // Hapus 'id_seni' dari sini
            'nama_lengkap' => $this->input->post('nama_lengkap'),
            'gelar_pendidikan' => $this->input->post('gelar_pendidikan'),
            'nama_alias' => $this->input->post('nama_alias'),
            'jenis_kelamin' => $this->input->post('jenis_kelamin'),
            'tempat_lahir' => $this->input->post('tempat_lahir'),
            'tanggal_lahir' => $this->input->post('tanggal_lahir'),
            'kewarganegaraan' => $this->input->post('kewarganegaraan'),
            'agama' => $this->input->post('agama'),
            'jenis_identitas' => $this->input->post('jenis_identitas'),
            'nama_ayah' => $this->input->post('nama_ayah'),
            'nama_ibu' => $this->input->post('nama_ibu'),
            'foto_identitas' => $foto_identitas,
            'pas_foto' => $pas_foto,
            'jenis_sdm' => 'perorangan'
        ];
        $id_sdm = $this->Pendaftaran_model->create_sdm($data_sdm);

        if (!$id_sdm) {
            throw new Exception("Gagal menyimpan data sdm.");
        }

        // Simpan data jenis seni ke tabel 'sdm_seni'
        foreach ($jenis_seni_array as $id_seni) {
            $data_sdm_seni = [
                'id_sdm' => $id_sdm,
                'id_seni' => $id_seni
            ];
            $this->Pendaftaran_model->save_sdm_seni($data_sdm_seni);
        }

        // PERBAIKAN: Hasilkan dan simpan NRK setelah mendapatkan id_sdm
        $nrk = $this->Pendaftaran_model->generate_nrk_perorangan($this->input->post('nik'), $id_sdm);
        $this->Pendaftaran_model->update_nrk_perorangan($id_sdm, $nrk);

        // Simpan data alamat dan kontak
        $data_alamat_kontak = [
            'id_sdm' => $id_sdm,
            'provinsi' => $this->input->post('provinsi'),
            'kabupaten' => $this->input->post('kabupaten'),
            'kecamatan' => $this->input->post('kecamatan'),
            'desa' => $this->input->post('desa'),
            'alamat' => $this->input->post('alamat'),
            'no_hp' => $this->input->post('no_hp'),
            'email' => $this->input->post('email'),
            'website' => $this->input->post('website'),
            'facebook' => $this->input->post('facebook'),
            'twitter' => $this->input->post('twitter'),
            'instagram' => $this->input->post('instagram'),
            'youtube' => $this->input->post('youtube'),
        ];
        $this->Pendaftaran_model->create_sdm_alamat_kontak($data_alamat_kontak);
        
        // Simpan data riwayat
        $this->simpan_riwayat($id_sdm);
        
        $this->db->trans_commit();
        $this->session->set_flashdata('success', 'Pendaftaran berhasil!');
        redirect('pendaftaran/terimakasih');

    } catch (Exception $e) {
        $this->db->trans_rollback();
        // Hapus file yang sudah terlanjur diunggah jika terjadi rollback
        if (isset($foto_identitas)) {
            @unlink('./uploads/foto_sdm/' . $foto_identitas);
        }
        if (isset($pas_foto)) {
            @unlink('./uploads/foto_sdm/' . $pas_foto);
        }
        $this->session->set_flashdata('error', 'Pendaftaran gagal: ' . $e->getMessage());
        $this->form_perorangan();
    }
}
private function upload_file($input_name, $folder) {
    // Cek apakah ada file yang diunggah
    if (empty($_FILES[$input_name]['name'])) {
        return NULL;
    }

    $config['upload_path'] = './assets/uploads/' . $folder . '/';
    $config['allowed_types'] = 'jpg|png|jpeg|pdf|doc|docx';
    $config['max_size'] = 2048;
    $config['file_name'] = $folder . '_' . uniqid() . '_' . time();

    if (!is_dir($config['upload_path'])) {
        mkdir($config['upload_path'], 0777, TRUE);
    }

    // Hanya perlu inisialisasi sekali per panggilan fungsi
    $this->load->library('upload', $config);
    
    if ($this->upload->do_upload($input_name)) {
        return $this->upload->data('file_name');
    }
    
    // Jika upload gagal, lemparkan exception agar transaksi bisa di-rollback
    throw new Exception("Gagal mengunggah file " . $this->upload->display_errors('', ''));
}
    
    private function simpan_riwayat($id_sdm) {
        // Riwayat Pendidikan
        $pendidikan = $this->input->post('pendidikan');
        if (is_array($pendidikan)) {
            foreach ($pendidikan as $data) {
                $data['id_sdm'] = $id_sdm;
                $this->Pendaftaran_model->create_sdm_pendidikan($data);
            }
        }
        
        // Riwayat Pelatihan
        $pelatihan = $this->input->post('pelatihan');
        if (is_array($pelatihan)) {
            foreach ($pelatihan as $data) {
                $data['id_sdm'] = $id_sdm;
                $this->Pendaftaran_model->create_sdm_pelatihan($data);
            }
        }

        // Riwayat Pekerjaan
        $pekerjaan = $this->input->post('pekerjaan');
        if (is_array($pekerjaan)) {
            foreach ($pekerjaan as $data) {
                $data['id_sdm'] = $id_sdm;
                $this->Pendaftaran_model->create_sdm_pekerjaan($data);
            }
        }

        // Riwayat Aktivitas Kebudayaan
        $aktivitas = $this->input->post('aktivitas');
        if (is_array($aktivitas)) {
            foreach ($aktivitas as $data) {
                $data['id_sdm'] = $id_sdm;
                $this->Pendaftaran_model->create_sdm_aktivitas($data);
            }
        }

        // Riwayat Penghargaan
        $penghargaan = $this->input->post('penghargaan');
        if (is_array($penghargaan)) {
            foreach ($penghargaan as $data) {
                $data['id_sdm'] = $id_sdm;
                $this->Pendaftaran_model->create_sdm_penghargaan($data);
            }
        }
    }
    // Pastikan fungsi ini ada di controller Anda (misalnya Pendaftaran.php)
        public function form_kelompok() {
            // Memuat model
            $this->load->model('Pendaftaran_model');

            // Mengambil data peran dari model
            $data['peran'] = $this->Pendaftaran_model->get_all_peran();

            // Mengambil data lain jika diperlukan
            $data['jenis_seni'] = $this->Pendaftaran_model->get_all_jenis_seni();
            $data['level'] = $this->Pendaftaran_model->get_all_level();
            
            // Memuat view dengan data
            $this->load->view('pendaftaran/form_kelompok', $data);
        }

    public function submit_kelompok() {
    // Validasi form
    $this->form_validation->set_rules('nama_sanggar', 'Nama Sanggar', 'required|is_unique[sanggar.nama_sanggar]');
    $this->form_validation->set_rules('provinsi', 'Provinsi', 'required');
    $this->form_validation->set_rules('kabupaten', 'Kabupaten/Kota', 'required');
    $this->form_validation->set_rules('no_hp', 'Nomor HP', 'required|min_length[7]|max_length[15]');
    $this->form_validation->set_rules('pengurus_nik[]', 'NIK Ketua', 'required|numeric|exact_length[16]');
    

    if ($this->form_validation->run() == FALSE) {
        $this->session->set_flashdata('error', validation_errors());
        $this->form_kelompok();
        return;
    }
    // LAKUKAN VALIDASI MANUAL UNTUK JENIS SENI
    $jenis_seni_array = $this->input->post('jenis_seni');
    if (empty($jenis_seni_array) || !is_array($jenis_seni_array)) {
        $this->session->set_flashdata('error', '<p>Jenis Seni wajib dipilih.</p>');
        $this->form_perorangan();
        return;
    }

    $this->db->trans_begin();
    try {
        // Upload berkas media
        $media_data = $this->upload_files_sanggar();

        // Data utama sanggar
        $data_sanggar = [
            'nama_sanggar' => $this->input->post('nama_sanggar'),
            'jenis_lembaga' => $this->input->post('jenis_lembaga'),
            'status_hukum' => $this->input->post('status_hukum'),
            'nomor_identitas' => $this->input->post('nomor_identitas'),
            'tanggal_berdiri' => $this->input->post('tanggal_berdiri'),
            'tujuan' => $this->input->post('tujuan'),
            'deskripsi' => $this->input->post('deskripsi'),
            'status_kepemilikan' => $this->input->post('status_kepemilikan'),
            'nama_pemilik' => $this->input->post('nama_pemilik'),
            'created_at' => date('Y-m-d H:i:s')
        ];
        $sanggar_id = $this->Pendaftaran_model->create_sanggar($data_sanggar);

        // Simpan Jenis Seni ke tabel relasi sanggar_seni
        foreach ($jenis_seni_array as $id_seni) {
            $data_sanggar_seni = [
                'id_sanggar' => $sanggar_id,
                'id_seni' => $id_seni
            ];
            $this->Pendaftaran_model->save_sanggar_seni($data_sanggar_seni);
        }

        // Hasilkan dan simpan NRK sanggar
        $pengurus_niks = $this->input->post('pengurus_nik');
        $ketua_nik = is_array($pengurus_niks) && isset($pengurus_niks[0]) ? $pengurus_niks[0] : null;
        if ($ketua_nik) {
            $nrk_sanggar = $this->Pendaftaran_model->generate_nrk_sanggar($ketua_nik, $sanggar_id);
            $this->Pendaftaran_model->update_nrk_sanggar($sanggar_id, $nrk_sanggar);
        }

        // Simpan data alamat & kontak
        $data_alamat_kontak = [
            'id_sanggar' => $sanggar_id,
            'provinsi' => $this->input->post('provinsi'),
            'kabupaten' => $this->input->post('kabupaten'),
            'kecamatan' => $this->input->post('kecamatan'),
            'desa' => $this->input->post('desa'),
            'alamat' => $this->input->post('alamat'),
            'latitude' => $this->input->post('latitude'),
            'longitude' => $this->input->post('longitude'),
            'no_hp' => $this->input->post('no_hp'),
            'email' => $this->input->post('alamat_email'),
            'website' => $this->input->post('alamat_website'),
            'facebook' => $this->input->post('akun_facebook'),
            'twitter' => $this->input->post('akun_twitter'),
            'instagram' => $this->input->post('akun_instagram'),
            'youtube' => $this->input->post('akun_youtube'),
        ];
        $this->Pendaftaran_model->create_sanggar_alamat_kontak($data_alamat_kontak);

        // Simpan data media dan legalitas
        $media_data['id_sanggar'] = $sanggar_id;
        $this->Pendaftaran_model->create_sanggar_media_legalitas($media_data);

        // Proses pengurus
        $pengurus_data_batch = [];
        $pengurus_peran_names = $this->input->post('pengurus_peran');
        $id_user_session = $this->session->userdata('id_user'); // 👈 Ambil ID user dari session
        
        if (is_array($pengurus_niks)) {
            foreach ($pengurus_niks as $key => $nik) {
                if (!empty($nik)) {
                    $sdm_exist = $this->Pendaftaran_model->get_sdm_by_nik($nik);
                    $id_sdm = 0;

                    if (!$sdm_exist) {
                        $data_sdm = [
                            'nik' => $nik,
                            'nama_lengkap' => $this->input->post('pengurus_nama')[$key],
                            'jenis_sdm' => 'pengurus',
                            'gelar_pendidikan' => $this->input->post('pengurus_gelar_pendidikan')[$key] ?? null,
                            'jenis_kelamin' => $this->input->post('pengurus_jk')[$key] ?? null,
                            'tempat_lahir' => $this->input->post('pengurus_tempat_lahir')[$key] ?? null,
                            'tanggal_lahir' => $this->input->post('pengurus_tanggal_lahir')[$key] ?? null,
                            'kewarganegaraan' => $this->input->post('pengurus_kewarganegaraan')[$key] ?? null,
                            'agama' => $this->input->post('pengurus_agama')[$key] ?? null,
                            'created_at' => date('Y-m-d H:i:s'),
                        ];
                        $id_sdm = $this->Pendaftaran_model->create_sdm($data_sdm);

                        $nrk_perorangan = $this->Pendaftaran_model->generate_nrk_perorangan($nik, $id_sdm);
                        $this->Pendaftaran_model->update_sdm_nrk($id_sdm, $nrk_perorangan);
                    } else {
                        $id_sdm = $sdm_exist->id_sdm;
                        if (empty($sdm_exist->nrk_perorangan)) {
                            $nrk_perorangan = $this->Pendaftaran_model->generate_nrk_perorangan($nik, $id_sdm);
                            $this->Pendaftaran_model->update_sdm_nrk($id_sdm, $nrk_perorangan);
                        }
                        $this->Pendaftaran_model->update_sdm_type($id_sdm, 'pengurus');
                    }

                    $id_peran = $this->Pendaftaran_model->get_peran_id($pengurus_peran_names[$key]);
                    $pengurus_data_batch[] = [
                        'id_sanggar' => $sanggar_id,
                        'id_sdm' => $id_sdm,
                        'id_peran' => $id_peran,
                        'id_user' => $id_user_session // 👈 Tambahkan baris ini
                    ];
                }
            }
        }
        $this->Pendaftaran_model->create_pengurus_sanggar($pengurus_data_batch);

        // Simpan riwayat kegiatan dan penghargaan
        $this->simpan_riwayat_kelompok($sanggar_id);

        // Proses anggota
        $anggota_data_batch = [];
        $anggota_niks = $this->input->post('anggota_nik');
        $anggota_level_names = $this->input->post('anggota_level');
        if (is_array($anggota_niks)) {
            foreach ($anggota_niks as $key => $nik) {
                if (!empty($nik)) {
                    $id_level = $this->Pendaftaran_model->get_level_id($anggota_level_names[$key]);
                    $anggota_data_batch[] = [
                        'id_sanggar' => $sanggar_id,
                        'nik' => $nik,
                        'nama_anggota' => $this->input->post('anggota_nama')[$key],
                        'jenis_kelamin' => $this->input->post('anggota_jk')[$key] ?? null,
                        'tempat_lahir' => $this->input->post('anggota_tempat_lahir')[$key] ?? null,
                        'tanggal_lahir' => $this->input->post('anggota_tanggal_lahir')[$key] ?? null,
                        'kewarganegaraan' => $this->input->post('anggota_kewarganegaraan')[$key] ?? null,
                        'agama' => $this->input->post('anggota_agama')[$key] ?? null,
                        'nomor_hp' => $this->input->post('anggota_hp')[$key] ?? null,
                        'id_level' => $id_level,
                    ];
                }
            }
        }
        $this->Pendaftaran_model->create_anggota_kelompok($anggota_data_batch);

        $this->db->trans_commit();
        $this->session->set_flashdata('success', 'Pendaftaran sanggar/kelompok berhasil!');
        redirect('pendaftaran/terimakasih/' . $sanggar_id);

    } catch (Exception $e) {
        $this->db->trans_rollback();
        $this->session->set_flashdata('error', 'Pendaftaran gagal: ' . $e->getMessage());
        $this->form_kelompok();
    }
}

    private function upload_files_sanggar() {
    $files = [
        'logo',
        'foto_papan_nama',
        'foto_bangunan_dalam',
        'foto_bangunan_luar',
        'dokumen_legalitas'
    ];
    $uploaded_data = [];
    foreach ($files as $name) {
        if (isset($_FILES[$name]) && $_FILES[$name]['error'] == 0) {
            // Langsung panggil upload_file dengan folder 'file_sangga'
            $uploaded_data[$name] = $this->upload_file($name, 'file_sanggar');
        }
    }
    return $uploaded_data;
}
    
    private function simpan_riwayat_kelompok($id_sanggar) {
    // ...
    // Riwayat Penghargaan
    $penghargaan = $this->input->post('nama_penghargaan');
    if (is_array($penghargaan)) {
        $data_batch = [];
        foreach ($penghargaan as $key => $val) {
            if (!empty($val)) {
                // Ganti 'bukti_penghargaan_sanggar' dengan 'file_sangga'
                $bukti = $this->upload_file_array('bukti_penghargaan', $key, 'file_sanggar');
                $data_batch[] = [
                    'id_sanggar' => $id_sanggar,
                    'nama_penghargaan' => $val,
                    'pemberi_penghargaan' => $this->input->post('pemberi_penghargaan')[$key],
                    'tahun' => $this->input->post('tahun_penghargaan')[$key],
                    'tingkat' => $this->input->post('tingkat_penghargaan')[$key],
                    'bukti' => $bukti,
                ];
            }
        }
        if (!empty($data_batch)) {
            $this->Pendaftaran_model->create_sanggar_penghargaan($data_batch);
        }
    }
}

    private function upload_file_array($input_name, $index, $folder) {
    if (!isset($_FILES[$input_name]['name'][$index]) || $_FILES[$input_name]['name'][$index] == '') {
        return NULL;
    }

    $_FILES['userfile']['name']     = $_FILES[$input_name]['name'][$index];
    $_FILES['userfile']['type']     = $_FILES[$input_name]['type'][$index];
    $_FILES['userfile']['tmp_name'] = $_FILES[$input_name]['tmp_name'][$index];
    $_FILES['userfile']['error']    = $_FILES[$input_name]['error'][$index];
    $_FILES['userfile']['size']     = $_FILES[$input_name]['size'][$index];

    // Perbaiki baris ini
    $config['upload_path'] = './uploads/' . $folder . '/';
    $config['allowed_types'] = 'jpg|png|jpeg|pdf|doc|docx';
    $config['max_size'] = 2048;

    if (!is_dir($config['upload_path'])) {
        mkdir($config['upload_path'], 0777, TRUE);
    }

    $this->load->library('upload', $config);
    $this->upload->initialize($config);
    
    if ($this->upload->do_upload('userfile')) {
        return $this->upload->data('file_name');
    }
    return NULL;
}
    // application/controllers/Pendaftaran.php

public function terimakasih($id = null) {
    // Check if the ID is provided.
    // This prevents errors in the view if no ID is passed.
    if ($id === null) {
        // You can redirect to another page or show a 404 error here
        // For example:
        // redirect('pendaftaran/dashboard'); 
        
        // Or just set a default value to prevent the view error
        $data['id'] = null;

    } else {
        // If the ID is provided, pass it to the view
        $data['id'] = $id;
    }

    $this->load->view('form_terimakasih', $data);
}


    public function cetak_bukti($id = null) {
    $user_id = $this->session->userdata('id_user');
    $user_nik = $this->session->userdata('nik');

    if (empty($user_nik) && $user_id) {
        $user = $this->db->get_where('users', ['id_user' => $user_id])->row();
        $user_nik = $user ? $user->nik : null;
    }

    $data = [
        'sanggar'   => null,
        'pendaftar' => null,
        'pengurus'  => [],
        'anggota'   => [],
        'jenis_seni_list' => []
    ];

    // =============== CARI DATA SANGGAR ===============
    $this->db->select('s.*, sak.alamat');
    $this->db->from('sanggar s');
    $this->db->join('pengurus_sanggar ps', 'ps.id_sanggar = s.id_sanggar');
    $this->db->join('sdm', 'sdm.id_sdm = ps.id_sdm');
    $this->db->join('sanggar_alamat_kontak sak', 'sak.id_sanggar = s.id_sanggar', 'left');

    if ($id) {
        $this->db->where('s.id_sanggar', $id);
    } else {
        $this->db->where('sdm.nik', $user_nik);
    }

    $this->db->order_by('s.created_at', 'DESC');
    $this->db->limit(1);
    $data_sanggar = $this->db->get()->row();

    if ($data_sanggar) {
        $data['sanggar'] = $data_sanggar;
        $sanggar_id = $data_sanggar->id_sanggar;
        $data['jenis_seni_list'] = $this->Pendaftaran_model->get_seni_sanggar($sanggar_id);

        // Pengurus
        $this->db->select('ps.*, sdm.nama_lengkap, sdm.nik, sdm.jenis_kelamin, 
                          sdm.nrk_perorangan, mp.nama_peran, sak.no_hp');
        $this->db->from('pengurus_sanggar ps');
        $this->db->join('sdm', 'sdm.id_sdm = ps.id_sdm', 'left');
        $this->db->join('master_peran mp', 'mp.id_peran = ps.id_peran', 'left');
        $this->db->join('sdm_alamat_kontak sak', 'sak.id_sdm = sdm.id_sdm', 'left');
        $this->db->where('ps.id_sanggar', $sanggar_id);
        $data['pengurus'] = $this->db->get()->result();

        // Anggota
        $this->db->select('ak.*, ml.nama_level');
        $this->db->from('anggota_kelompok ak');
        $this->db->join('master_level ml', 'ml.id_level = ak.id_level', 'left');
        $this->db->where('ak.id_sanggar', $sanggar_id);
        $data['anggota'] = $this->db->get()->result();

    } else {
        // ... (Bagian perorangan) ...
        $this->db->select('sdm.*, sak.alamat, sak.no_hp');
        $this->db->from('sdm');
        $this->db->join('sdm_alamat_kontak sak', 'sak.id_sdm = sdm.id_sdm', 'left');

        if ($id) {
            $this->db->where('sdm.id_sdm', $id);
        } else {
            $this->db->where('sdm.nik', $user_nik);
        }

        $this->db->order_by('sdm.created_at', 'DESC');
        $this->db->limit(1);
        $data_pendaftar = $this->db->get()->row();

        if ($data_pendaftar) {
            $data['pendaftar'] = $data_pendaftar;
            // TAMBAHKAN KODE INI untuk mengambil jenis seni perorangan
            $data['jenis_seni_list'] = $this->Pendaftaran_model->get_jenis_seni_by_sdm_id($data_pendaftar->id_sdm);
        }
    }

    if ($data['sanggar'] || $data['pendaftar']) {
        $this->load->view('pendaftaran/bukti_pendaftaran', $data);
    } else {
        $this->session->set_flashdata('error', 'Data pendaftaran tidak ditemukan.');
        redirect('pendaftaran');
    }
}


    public function dashboard() 
{
    $id_user = $this->session->userdata('id_user');
    $user    = $this->Pendaftaran_model->get_user_by_id($id_user);

    if (!$user) redirect('auth/login');

    if ($user->id_role == 2) {
        // perorangan
       $data['user']      = $user;
        $data['pendaftar'] = $this->Pendaftaran_model->get_sdm_by_nik_lengkap($user->nik);

        if ($data['pendaftar']) {
            $data['seni'] = $this->Pendaftaran_model->get_sdm_seni($data['pendaftar']->id_sdm);
        } else {
            $data['seni'] = [];
        }

        $this->load->view('pendaftaran/dashboard_perorangan', $data);

    } elseif ($user->id_role == 3) {
    // kelompok
    $data['user']    = $user;
    $sanggar         = $this->Pendaftaran_model->get_sanggar_by_user($user->id_user);
    $data['sanggar'] = $sanggar;

    if ($sanggar) {
        $data['seni']    = $this->Pendaftaran_model->get_seni_sanggar($sanggar->id_sanggar);
        $data['anggota'] = $this->Pendaftaran_model->get_anggota_by_sanggar($sanggar->id_sanggar);
        $data['pengurus']= $this->Pendaftaran_model->get_pengurus_by_sanggar_lengkap($sanggar->id_sanggar);
    } else {
        $data['seni']    = [];
        $data['anggota'] = [];
        $data['pengurus']= [];
    }

    $this->load->view('pendaftaran/dashboard_kelompok', $data);
}

}

    public function profile() {
    $id_user = $this->session->userdata('id_user');
    $user = $this->Pendaftaran_model->get_user_by_id($id_user);
    $data['user'] = $user;
    $data['jenis_pendaftar'] = null;
    $data['pendaftar'] = null;
    $data['sanggar'] = null;

    if ($user) {
        // PERBAIKAN: Gunakan id_role untuk menentukan jenis pendaftar
        if ($user->id_role == 3) { // Role Kelompok
            $data['jenis_pendaftar'] = 'kelompok';
            $data['sanggar'] = $this->Pendaftaran_model->get_sanggar_by_pengurus_nik($user->nik);
        } else { // Role Perorangan (role 2)
            $data['jenis_pendaftar'] = 'perorangan';
            // Perbaikan: Pastikan mengambil data sdm
            $data['pendaftar'] = $this->Pendaftaran_model->get_sdm_by_nik($user->nik);
        }
    }
    $this->load->view('pendaftaran/profile', $data);
}

    // PENTING: ini adalah fungsi controller 'Pendaftaran.php'
public function biodata() {
    $id_user = $this->session->userdata('id_user');
    if (!$id_user) {
        redirect('auth/login');
    }

    $user = $this->Pendaftaran_model->get_user_by_id($id_user);
    if (!$user) {
        redirect('auth/login');
    }

    if ($user->id_role == 3) { 
        // 👉 kelompok
        $data['user'] = $user;
        $sanggar = $this->Pendaftaran_model->get_sanggar_by_user($id_user);
        $data['sanggar'] = $sanggar;
        
        // Cek apakah sanggar ditemukan sebelum mencoba mengambil data terkait
        if ($sanggar) {
            $data['anggota']  = $this->Pendaftaran_model->get_anggota_by_sanggar($sanggar->id_sanggar);
            $data['pengurus'] = $this->Pendaftaran_model->get_pengurus_by_sanggar_lengkap($sanggar->id_sanggar);
            $data['seni']     = $this->Pendaftaran_model->get_seni_sanggar($sanggar->id_sanggar);
        } else {
            // Jika sanggar tidak ditemukan, inisialisasi variabel dengan array kosong
            $data['anggota']  = [];
            $data['pengurus'] = [];
            $data['seni']     = [];
        }

        $this->load->view('pendaftaran/biodata_kelompok', $data);
    } elseif ($user->id_role == 2) { 
    // 👉 perorangan
    $data['user']     = $user;
    $pendaftar        = $this->Pendaftaran_model->get_sdm_by_user($id_user);
    $data['pendaftar']= $pendaftar;

    if ($pendaftar) {
        $data['seni']       = $this->Pendaftaran_model->get_seni_by_sdm($pendaftar->id_sdm);
        // var_dump($data['seni']); 
        //     die(); 
        $data['pendidikan'] = $this->Pendaftaran_model->get_riwayat_pendidikan_by_sdm($pendaftar->id_sdm);
        $data['pelatihan']  = $this->Pendaftaran_model->get_riwayat_pelatihan_by_sdm($pendaftar->id_sdm);
        $data['pekerjaan']  = $this->Pendaftaran_model->get_riwayat_pekerjaan_by_sdm($pendaftar->id_sdm);
        $data['aktivitas']  = $this->Pendaftaran_model->get_riwayat_aktivitas_by_sdm($pendaftar->id_sdm);
        $data['penghargaan']= $this->Pendaftaran_model->get_riwayat_penghargaan_by_sdm($pendaftar->id_sdm);
    } else {
        $data['seni']       = [];
        $data['pendidikan'] = [];
        $data['pelatihan']  = [];
        $data['pekerjaan']  = [];
        $data['aktivitas']  = [];
        $data['penghargaan']= [];
    }
    $this->load->view('pendaftaran/biodata_perorangan', $data);
}

}
// Tambahan di Pendaftaran_model.php
// Pastikan fungsi ini ada dan benar
public function get_riwayat_aktivitas_by_sdm($id_sdm) {
    $this->db->select('riwayat_aktivitas.*, jenis_seni.nama_seni');
    $this->db->from('riwayat_aktivitas');
    $this->db->join('jenis_seni', 'jenis_seni.id_seni = riwayat_aktivitas.id_seni', 'left');
    $this->db->where('riwayat_aktivitas.id_sdm', $id_sdm);
    return $this->db->get()->result();
}
    
    // =======================
    // EDIT & HAPUS PENDAFTAR
    // =======================
    public function edit_pendaftar($id) {
        if ($this->input->method() === 'post') {
            $data = [
                'nama' => $this->input->post('nama'),
                'alamat' => $this->input->post('alamat'),
                'no_hp' => $this->input->post('no_hp'),
                'jenis_kelamin' => $this->input->post('jenis_kelamin'),
                'id_seni' => $this->input->post('id_seni'),
                'deskripsi' => $this->input->post('deskripsi'),
            ];
            $this->Pendaftaran_model->update_pendaftar($id, $data);
            $this->session->set_flashdata('success', 'Data pendaftar berhasil diupdate.');
            redirect('pendaftaran/biodata');
        } else {
            $data['pendaftar'] = $this->Pendaftaran_model->get_pendaftar_by_id($id);
            if (!$data['pendaftar']) {
                show_404();
            }
            $data['jenis_seni'] = $this->Pendaftaran_model->get_all_jenis_seni();
            $this->load->view('pendaftaran/edit_pendaftar', $data);
        }
    }
    public function update_pendaftar($id)
{
    $this->load->library('form_validation');
    $this->load->library('upload'); // Pastikan library upload di-load

    $this->form_validation->set_rules('nama_lengkap', 'Nama Lengkap', 'required');
    $this->form_validation->set_rules('jenis_kelamin', 'Jenis Kelamin', 'required');
    $this->form_validation->set_rules('tanggal_lahir', 'Tanggal Lahir', 'required');

    if ($this->form_validation->run() == FALSE) {
        $this->edit_pendaftar($id);
    } else {
        // Data dasar untuk update
        $data_update = [
            'nama_lengkap'      => $this->input->post('nama_lengkap'),
            'gelar_pendidikan'  => $this->input->post('gelar_pendidikan'),
            'jenis_kelamin'     => $this->input->post('jenis_kelamin'),
            'tempat_lahir'      => $this->input->post('tempat_lahir'),
            'tanggal_lahir'     => $this->input->post('tanggal_lahir')
        ];

        // Cek apakah ada file foto yang diunggah
        if (!empty($_FILES['pas_foto']['name'])) {
            $config['upload_path']   = './assets/uploads/foto_sdm/';
            $config['allowed_types'] = 'jpg|jpeg|png';
            $config['max_size']      = 2048; // 2 MB
            $config['encrypt_name']  = TRUE; // Enkripsi nama file

            $this->upload->initialize($config);

            if ($this->upload->do_upload('pas_foto')) {
                // Upload berhasil
                $upload_data = $this->upload->data();
                $nama_file_baru = $upload_data['file_name'];

                // Hapus foto lama jika ada
                $pendaftar_lama = $this->Pendaftaran_model->get_pendaftar_by_id($id);
                if ($pendaftar_lama && !empty($pendaftar_lama->pas_foto)) {
                    $file_path = FCPATH . 'assets/uploads/foto_sdm/' . $pendaftar_lama->pas_foto;
                    if (file_exists($file_path)) {
                        unlink($file_path);
                    }
                }

                // Tambahkan nama file baru ke data update
                $data_update['pas_foto'] = $nama_file_baru;
                
            } else {
                // Upload gagal, tampilkan error dan kembali ke form
                $error = $this->upload->display_errors();
                $this->session->set_flashdata('error', 'Gagal mengunggah foto: ' . $error);
                redirect('pendaftaran/edit_pendaftar/' . $id);
                return;
            }
        }

        // Lanjutkan proses update ke model
        $this->Pendaftaran_model->update_pendaftar($id, $data_update);

        $this->session->set_flashdata('success', 'Data diri berhasil diperbarui ✅');
        redirect('pendaftaran/biodata');
    }
}

    public function hapus_pendaftar($id) {
        $this->Pendaftaran_model->delete_pendaftar($id);
        $this->session->set_flashdata('success', 'Biodata berhasil dihapus.');
        redirect('pendaftaran/biodata');
    }

    // =======================
    // EDIT SENI PERORANGAN
    // =======================
    public function edit_seni($id_seni_sdm) {
    // 1. Ambil data seni yang akan diedit dari tabel sdm_seni
    $seni_yang_diedit = $this->db->get_where('sdm_seni', ['id' => $id_seni_sdm])->row();

    // 2. Jika data tidak ditemukan, arahkan kembali dengan pesan kesalahan
    if (!$seni_yang_diedit) {
        $this->session->set_flashdata('error', 'Data seni tidak ditemukan.');
        redirect('pendaftaran/biodata');
        return;
    }

    // 3. Ambil semua jenis seni untuk mengisi dropdown
    $data['jenis_seni'] = $this->db->get('jenis_seni')->result();

    // 4. Siapkan data seni yang akan diedit untuk view
    $data['seni'] = $seni_yang_diedit;

    // 5. Muat tampilan (view) edit seni.
    // Tidak ada logika POST di sini.
    $this->load->view('pendaftaran/edit_seni', $data);
}

public function update_seni_perorangan() {
    // 1. Ambil ID unik dari input hidden di form
    $id_seni_sdm = $this->input->post('id_seni_sdm'); 

    // 2. Lakukan validasi dasar untuk memastikan ID ada
    if (!$id_seni_sdm) {
        $this->session->set_flashdata('error', 'ID data tidak valid. Silakan coba lagi.');
        redirect('pendaftaran/biodata');
        return;
    }

    // 3. Ambil data yang dikirim dari form
    $update_data = [
        'id_seni'   => $this->input->post('id_seni'),
        'deskripsi' => $this->input->post('deskripsi')
    ];
    
    // 4. Lakukan pembaruan data di database
    // Gunakan ID unik dari tabel sdm_seni sebagai kondisi WHERE
    $this->db->where('id', $id_seni_sdm)->update('sdm_seni', $update_data);
    
    // 5. Periksa apakah operasi update berhasil
    if ($this->db->affected_rows() > 0) {
        $this->session->set_flashdata('success', 'Data seni berhasil diperbarui!');
    } else {
        $this->session->set_flashdata('error', 'Gagal memperbarui data atau tidak ada perubahan.');
    }

    // 6. Alihkan kembali ke halaman biodata
    redirect('pendaftaran/biodata');
}


    // =======================
    // HAPUS SENI SANGGAR
    // =======================
    public function hapus_seni_sanggar($id_sanggar, $id_seni) {
    // Validasi input
    if (!$id_sanggar || !$id_seni) {
        $this->session->set_flashdata('error', 'ID sanggar atau ID seni tidak valid.');
        redirect('pendaftaran/biodata');
        return;
    }

    // Ambil user ID dan cek kepemilikan
    $user_id = $this->session->userdata('id_user');
    $sanggar = $this->Pendaftaran_model->get_sanggar_by_user($user_id);

    // Pastikan user adalah pemilik sanggar yang valid
    if (!$sanggar || $sanggar->id_sanggar != $id_sanggar) {
        $this->session->set_flashdata('error', 'Anda tidak memiliki hak untuk menghapus data ini.');
        redirect('pendaftaran/biodata');
        return;
    }

    // Panggil model untuk menghapus data dari tabel relasi
    $hapus = $this->Pendaftaran_model->hapus_seni_sanggar($id_sanggar, $id_seni);

    if ($hapus) {
        $this->session->set_flashdata('success', 'Jenis seni berhasil dihapus dari sanggar.');
    } else {
        $this->session->set_flashdata('error', 'Gagal menghapus jenis seni.');
    }

    redirect('pendaftaran/biodata');
}

    // =======================
    // EDIT & HAPUS ANGGOTA
    // =======================
    public function edit_anggota($id_anggota) {
    // Gabungkan data dari tabel anggota_kelompok dengan master_level
    $this->db->select('anggota_kelompok.*, master_level.nama_level');
    $this->db->from('anggota_kelompok');
    $this->db->join('master_level', 'anggota_kelompok.id_level = master_level.id_level', 'left');
    $this->db->where('anggota_kelompok.id_anggota', $id_anggota);
    $anggota = $this->db->get()->row();
    
    if (!$anggota) {
        $this->session->set_flashdata('error', 'Data anggota tidak ditemukan.');
        redirect('pendaftaran/biodata');
    }

    $data['anggota'] = $anggota;

    // Ambil data master level untuk dropdown
    $data['master_level'] = $this->db->get('master_level')->result();

    if ($this->input->post()) {
        $new_status = $this->input->post('status', true);
        $update = [
            'id_level' => $this->input->post('id_level', true),
            'status' => $new_status,
        ];
        
        // Cek jika status berubah dari aktif ke nonaktif
        if ($anggota->status == 'aktif' && $new_status == 'nonaktif') {
            $update['tanggal_nonaktif'] = date('Y-m-d');
            
            // Catat riwayat keanggotaan
            $riwayat_data = [
                'id_sanggar' => $anggota->id_sanggar,
                'id_anggota' => $anggota->id_anggota,
                'tanggal_daftar' => $anggota->tanggal_daftar,
                'tanggal_nonaktif' => date('Y-m-d'),
                'status' => 'nonaktif'
            ];
            $this->db->insert('riwayat_keanggotaan', $riwayat_data);
        }

        $this->db->where('id_anggota', $id_anggota)->update('anggota_kelompok', $update);
        
        if ($this->db->affected_rows() > 0) {
            $this->session->set_flashdata('success', 'Data anggota berhasil diperbarui!');
        } else {
            $this->session->set_flashdata('error', 'Tidak ada perubahan atau gagal update.');
        }
        redirect('pendaftaran/biodata');
    } else {
        $this->load->view('pendaftaran/edit_anggota', $data);
    }
}

    public function hapus_anggota($id) {
        $this->Pendaftaran_model->delete_anggota($id);
        $this->session->set_flashdata('success', 'Anggota berhasil dihapus.');
        redirect('pendaftaran/biodata');
    }

    // =======================
    // EDIT PENGURUS
    // =======================
    public function edit_pengurus($id_pengurus)
{
    // 1. Memeriksa keberadaan pengurus berdasarkan PRIMARY KEY (id_pengurus)
    $pengurus = $this->db->get_where('pengurus_sanggar', ['id_pengurus' => $id_pengurus])->row();
    if (!$pengurus) {
        $this->session->set_flashdata('error', 'Data pengurus tidak ditemukan.');
        redirect('pendaftaran/biodata');
    }

    // 2. Mengambil data personal SDM yang terhubung menggunakan id_sdm
    $sdm = $this->db->get_where('sdm', ['id_sdm' => $pengurus->id_sdm])->row();

    // 3. Mengambil data master peran untuk dropdown
    $master_peran = $this->db->get('master_peran')->result();

    $data['pengurus'] = $pengurus;
    $data['sdm'] = $sdm; // Menambahkan data sdm ke view
    $data['master_peran'] = $master_peran; // Menambahkan data master_peran ke view

    // 4. Memproses form submission
    if ($this->input->post()) {
        // Data yang di-update hanya untuk tabel pengurus_sanggar
        $update_pengurus = [
            'id_peran' => $this->input->post('id_peran', true),
            'status' => $this->input->post('status', true),
        ];
        
        $this->db->where('id_pengurus', $id_pengurus)->update('pengurus_sanggar', $update_pengurus);
        
        if ($this->db->affected_rows() > 0) {
            $this->session->set_flashdata('success', 'Data pengurus berhasil diperbarui!');
        } else {
            $this->session->set_flashdata('error', 'Tidak ada perubahan atau gagal update.');
        }
        redirect('pendaftaran/biodata');
    } else {
        // 5. Menampilkan view edit
        $this->load->view('pendaftaran/edit_pengurus', $data);
    }
}

    // =======================
    // HAPUS PENGURUS
    // =======================
    public function hapus_pengurus($id_pengurus) {
        $pengurus = $this->db->where('id_pengurus', $id_pengurus)->get('pengurus_sanggar')->row();
        if (!$pengurus) {
            $this->session->set_flashdata('error', 'Data pengurus tidak ditemukan.');
            redirect('pendaftaran/biodata');
        }

        $this->db->where('id_pengurus', $id_pengurus)->delete('pengurus_sanggar');
        if ($this->db->affected_rows() > 0) {
            $this->session->set_flashdata('success', 'Data pengurus berhasil dihapus!');
        } else {
            $this->session->set_flashdata('error', 'Gagal menghapus data pengurus.');
        }
        redirect('pendaftaran/biodata');
    }

    // =======================
    // TAMBAH PENGURUS
    // =======================
    public function tambah_pengurus($id_sanggar) {
    // 1. Memeriksa keberadaan sanggar
    $sanggar = $this->db->get_where('sanggar', ['id_sanggar' => $id_sanggar])->row();
    if (!$sanggar) {
        $this->session->set_flashdata('error', 'Data sanggar tidak ditemukan.');
        redirect('pendaftaran/biodata');
    }

    $data['sanggar'] = $sanggar;

    // 2. Memproses data dari form (jika ada POST request)
    if ($this->input->post()) {
        $nik = $this->input->post('nik', true);
        
        // Memulai transaksi database
        $this->db->trans_start();
        
        // 3. Mencari id_sdm dan id_peran
        $sdm = $this->db->get_where('sdm', ['nik' => $nik])->row();
        $peran = $this->db->get_where('master_peran', ['nama_peran' => $this->input->post('nama_peran', true)])->row();

        // 4. Jika data SDM belum ada, buat record baru
        if (!$sdm) {
            $insert_sdm = [
                'nik' => $nik,
                'nama_lengkap' => $this->input->post('nama_lengkap', true),
                'jenis_kelamin' => $this->input->post('jenis_kelamin', true),
                'tempat_lahir' => $this->input->post('tempat_lahir', true),
                'tanggal_lahir' => $this->input->post('tanggal_lahir', true),
                'gelar_pendidikan' => $this->input->post('gelar_pendidikan', true),
                'kewarganegaraan' => $this->input->post('kewarganegaraan', true),
                'agama' => $this->input->post('agama', true),
            ];
            $this->db->insert('sdm', $insert_sdm);
            $id_sdm_baru = $this->db->insert_id();

            // Insert data kontak ke tabel sdm_alamat_kontak
            $insert_kontak = [
                'id_sdm' => $id_sdm_baru,
                'no_hp' => $this->input->post('no_hp', true),
            ];
            $this->db->insert('sdm_alamat_kontak', $insert_kontak);
            
            // Menggunakan id_sdm yang baru dibuat
            $id_sdm_final = $id_sdm_baru;

        } else {
            // Jika data SDM sudah ada, gunakan id_sdm yang sudah ada
            $id_sdm_final = $sdm->id_sdm;
            // Perbarui data kontak jika ada perubahan
            $this->db->where('id_sdm', $id_sdm_final)->update('sdm_alamat_kontak', ['no_hp' => $this->input->post('no_hp', true)]);
        }

        // 5. Memeriksa duplikasi pengurus di sanggar yang sama
        $pengurus_exist = $this->db->get_where('pengurus_sanggar', ['id_sanggar' => $id_sanggar, 'id_sdm' => $id_sdm_final])->row();
        if ($pengurus_exist) {
            $this->session->set_flashdata('error', 'NIK ' . $nik . ' sudah terdaftar sebagai pengurus di sanggar ini.');
            $this->db->trans_rollback();
            redirect('pendaftaran/tambah_pengurus/' . $id_sanggar);
        }

        // 6. Menyiapkan data untuk insert ke tabel pengurus_sanggar
        $insert_pengurus = [
            'id_sanggar' => $id_sanggar,
            'id_sdm' => $id_sdm_final,
            'id_peran' => $peran->id_peran,
            'status' => 'aktif', // Atur status default aktif
        ];
        
        // 7. Melakukan insert ke database
        $this->db->insert('pengurus_sanggar', $insert_pengurus);

        // Menyelesaikan transaksi
        $this->db->trans_complete();

        // 8. Memberikan notifikasi berdasarkan status transaksi
        if ($this->db->trans_status() === false) {
            $this->session->set_flashdata('error', 'Gagal menambahkan pengurus. Terjadi kesalahan pada database.');
        } else {
            $this->session->set_flashdata('success', 'Pengurus berhasil ditambahkan!');
        }

        redirect('pendaftaran/biodata');
    } else {
        // 9. Menampilkan form
        $this->load->view('pendaftaran/tambah_pengurus', $data);
    }
}

    // =======================
    // TAMBAH ANGGOTA
    // =======================
    public function tambah_anggota($id_sanggar) {
    // Cari data sanggar terlebih dahulu
    $sanggar = $this->db->where('id_sanggar', $id_sanggar)->get('sanggar')->row();
    if (!$sanggar) {
        $this->session->set_flashdata('error', 'Data sanggar tidak ditemukan.');
        redirect('pendaftaran/biodata');
        return;
    }
    
    // Siapkan data untuk view, ini harus selalu ada
    $data['sanggar'] = $sanggar;
    $data['master_level'] = $this->db->get('master_level')->result();

    // Jika form disubmit (metode POST)
    if ($this->input->post()) {
        $nik = $this->input->post('nik', true);
        
        // Cek apakah NIK sudah terdaftar aktif di sanggar lain
        $anggota_aktif = $this->db->get_where('anggota_kelompok', ['nik' => $nik, 'status' => 'aktif'])->row();
        
        if ($anggota_aktif) {
            $this->session->set_flashdata('error', 'NIK ' . $nik . ' sudah terdaftar sebagai anggota aktif di sanggar lain. Mohon nonaktifkan statusnya terlebih dahulu.');
            redirect('pendaftaran/tambah_anggota/' . $id_sanggar);
            return;
        }

        // Cek apakah NIK sudah terdaftar di sanggar ini (meskipun nonaktif)
        $anggota_sanggar_ini = $this->db->get_where('anggota_kelompok', ['id_sanggar' => $id_sanggar, 'nik' => $nik])->row();
        if ($anggota_sanggar_ini) {
            $this->session->set_flashdata('error', 'NIK ' . $nik . ' sudah terdaftar sebagai anggota di sanggar ini (status: ' . $anggota_sanggar_ini->status . ').');
            redirect('pendaftaran/tambah_anggota/' . $id_sanggar);
            return;
        }

        $insert = [
            'id_sanggar' => $id_sanggar,
            'nik' => $nik,
            'nama_anggota' => $this->input->post('nama_anggota', true),
            'jenis_kelamin' => $this->input->post('jenis_kelamin', true),
            'tempat_lahir' => $this->input->post('tempat_lahir', true),
            'tanggal_lahir' => $this->input->post('tanggal_lahir', true),
            'kewarganegaraan' => $this->input->post('kewarganegaraan', true),
            'agama' => $this->input->post('agama', true),
            'nomor_hp' => $this->input->post('nomor_hp', true),
            'id_level' => $this->input->post('id_level', true),
            'status' => 'aktif',
            'tanggal_daftar' => date('Y-m-d')
        ];
        
        $this->db->insert('anggota_kelompok', $insert);

        if ($this->db->affected_rows() > 0) {
            $this->session->set_flashdata('success', 'Anggota berhasil ditambahkan!');
        } else {
            $this->session->set_flashdata('error', 'Gagal menambahkan anggota.');
        }

        redirect('pendaftaran/biodata');
    } else {
        // Tampilkan view dengan data yang sudah disiapkan
        $this->load->view('pendaftaran/tambah_anggota', $data);
    }
}

    public function cari_anggota_by_nik($nik) {
    $response = [];

    // Cari langsung di tabel anggota_kelompok
    $anggota_data = $this->db->get_where('anggota_kelompok', ['nik' => $nik])->row();

    if ($anggota_data) {
        // Jika ditemukan, gunakan data dari tabel anggota_kelompok
        $response['status'] = 'success';
        $response['anggota'] = [
            'nik' => $anggota_data->nik,
            'nama_anggota' => $anggota_data->nama_anggota,
            'jenis_kelamin' => $anggota_data->jenis_kelamin,
            'tempat_lahir' => $anggota_data->tempat_lahir,
            'tanggal_lahir' => $anggota_data->tanggal_lahir,
            'kewarganegaraan' => $anggota_data->kewarganegaraan,
            'agama' => $anggota_data->agama,
            'nomor_hp' => $anggota_data->nomor_hp
        ];
    } else {
        // Jika tidak ditemukan di tabel anggota_kelompok
        $response['status'] = 'error';
        $response['message'] = 'NIK tidak ditemukan di database.';
    }
    
    header('Content-Type: application/json');
    echo json_encode($response);
}
    
    public function tambah_anggota_dari_pencarian($id_sanggar) {
    // ...
    if ($this->input->post()) {
        $nik = $this->input->post('nik', true);
        $id_level = $this->input->post('id_level', true);

        // 1. Cek apakah NIK sudah terdaftar aktif di sanggar manapun
        $anggota_aktif = $this->db->get_where('anggota_kelompok', ['nik' => $nik, 'status' => 'aktif'])->row();
        
        // 2. Jika ditemukan, beri pesan error
        if ($anggota_aktif) {
            $this->session->set_flashdata('error', 'NIK ' . $nik . ' sudah terdaftar sebagai anggota aktif di sanggar lain. Mohon nonaktifkan statusnya terlebih dahulu.');
            redirect('pendaftaran/tambah_anggota/' . $id_sanggar);
            return;
        }

        // 3. (Opsional) Cek apakah NIK sudah terdaftar di sanggar ini
        $anggota_sanggar_ini = $this->db->get_where('anggota_kelompok', ['id_sanggar' => $id_sanggar, 'nik' => $nik])->row();
        if ($anggota_sanggar_ini) {
            $this->session->set_flashdata('error', 'NIK ' . $nik . ' sudah terdaftar sebagai anggota di sanggar ini (status: ' . $anggota_sanggar_ini->status . ').');
            redirect('pendaftaran/tambah_anggota/' . $id_sanggar);
            return;
        }

        // ... lanjutkan dengan proses insert jika semua pemeriksaan lolos ...
        $insert = [
            'id_sanggar' => $id_sanggar,
            'nik' => $nik,
            'nama_anggota' => $this->input->post('nama_anggota', true),
            'jenis_kelamin' => $this->input->post('jenis_kelamin', true),
            'tempat_lahir' => $this->input->post('tempat_lahir', true),
            'tanggal_lahir' => $this->input->post('tanggal_lahir', true),
            'kewarganegaraan' => $this->input->post('kewarganegaraan', true),
            'agama' => $this->input->post('agama', true),
            'nomor_hp' => $this->input->post('nomor_hp', true),
            'id_level' => $id_level,
            'status' => 'aktif',
            'tanggal_daftar' => date('Y-m-d')
        ];

        $this->db->insert('anggota_kelompok', $insert);

        if ($this->db->affected_rows() > 0) {
            $this->session->set_flashdata('success', 'Anggota berhasil ditambahkan!');
        } else {
            $this->session->set_flashdata('error', 'Gagal menambahkan anggota.');
        }

        redirect('pendaftaran/biodata');
    }
}


    // =======================
    // EDIT PROFIL SANGGAR
    // =======================
    public function edit_jenis_seni_single($id_sanggar, $id_seni) {
    // Validasi kepemilikan sanggar seperti biasa
    // ...

    $data['sanggar'] = $this->Pendaftaran_model->get_sanggar_by_user($this->session->userdata('id_user'));
    
    // Ambil jenis seni yang sedang diedit
    $data['seni_saat_ini'] = $this->Pendaftaran_model->get_seni_by_id($id_seni);

    // Ambil semua jenis seni untuk dropdown
    $data['list_seni'] = $this->db->get('jenis_seni')->result();

    $this->load->view('pendaftaran/edit_seni_sanggar', $data);
}

    // =======================
    // TAMBAH SENI PERORANGAN
    // =======================
    public function tambah_seni_perorangan() {
        $user_id = $this->session->userdata('id_user');
        $user = $this->Pendaftaran_model->get_user_by_id($user_id);
        
        if (!$user) {
            $this->session->set_flashdata('error', 'Data pengguna tidak valid.');
            redirect('pendaftaran/biodata');
        }

        $pendaftar_utama = $this->Pendaftaran_model->get_pendaftar_by_nik_unique($user->nik);
        
        if (!$pendaftar_utama) {
            $this->session->set_flashdata('error', 'Anda belum memiliki data pendaftaran perorangan.');
            redirect('pendaftaran/form_perorangan');
        }

        if ($this->input->post()) {
            $this->form_validation->set_rules('id_seni', 'Jenis Seni', 'required');
            $this->form_validation->set_rules('deskripsi', 'Deskripsi', 'required');
            
            if ($this->form_validation->run() == FALSE) {
                $data['pendaftar'] = $pendaftar_utama;
                $data['jenis_seni'] = $this->Pendaftaran_model->get_all_jenis_seni();
                $this->load->view('pendaftaran/tambah_seni_perorangan', $data);
                return;
            }

            $nik = $user->nik;
            $id_seni_baru = $this->input->post('id_seni');

            $seni_sudah_ada = $this->db->get_where('pendaftar', ['nik' => $nik, 'id_seni' => $id_seni_baru])->row();
            if ($seni_sudah_ada) {
                $this->session->set_flashdata('error', 'Jenis seni ini sudah didaftarkan.');
                redirect('pendaftaran/tambah_seni_perorangan');
                return;
            }
            
            $data_seni_baru = [
                'nik' => $pendaftar_utama->nik,
                'nama' => $pendaftar_utama->nama,
                'alamat' => $pendaftar_utama->alamat,
                'no_hp' => $pendaftar_utama->no_hp,
                'jenis_kelamin' => $pendaftar_utama->jenis_kelamin,
                'id_seni' => $id_seni_baru,
                'deskripsi' => $this->input->post('deskripsi', true),
            ];
            
            $this->Pendaftaran_model->create_pendaftar($data_seni_baru);
            $this->session->set_flashdata('success', 'Seni baru berhasil ditambahkan!');
            redirect('pendaftaran/biodata');
        } else {
            $data['pendaftar'] = $pendaftar_utama;
            $data['jenis_seni'] = $this->Pendaftaran_model->get_all_jenis_seni();
            $this->load->view('pendaftaran/tambah_seni_perorangan', $data);
        }
    }
    
    public function simpan_seni_perorangan() {
    $this->form_validation->set_rules('id_seni', 'Jenis Seni', 'required');
    
    if ($this->form_validation->run() == FALSE) {
        $this->tambah_seni_perorangan();
        return;
    }

    $id_sdm = $this->input->post('id_sdm');
    $id_seni = $this->input->post('id_seni');
    $deskripsi = $this->input->post('deskripsi', true);

    // cek apakah sudah ada
    $cek = $this->db->get_where('sdm_seni', [
        'id_sdm' => $id_sdm,
        'id_seni' => $id_seni
    ])->row();

    if ($cek) {
        $this->session->set_flashdata('error', 'Jenis seni ini sudah ditambahkan.');
        redirect('pendaftaran/tambah_seni_perorangan');
        return;
    }

    // simpan ke tabel relasi
    $this->db->insert('sdm_seni', [
        'id_sdm' => $id_sdm,
        'id_seni' => $id_seni,
        'deskripsi' => $deskripsi
    ]);

    $this->session->set_flashdata('success', 'Seni berhasil ditambahkan!');
    redirect('pendaftaran/biodata');
}

    // =======================
    // TAMBAH SENI KELOMPOK
    // =======================
    public function tambah_seni_sanggar($id_sanggar)
{
    $this->load->model('Pendaftaran_model');
    $sanggar = $this->Pendaftaran_model->get_sanggar_by_id($id_sanggar);

    if (!$sanggar) {
        $this->session->set_flashdata('error', 'Sanggar tidak ditemukan.');
        redirect('pendaftaran/biodata');
        return;
    }

    if ($this->input->post()) {
        $id_seni_baru = $this->input->post('id_seni');
        
        // Data yang akan disimpan ke tabel sanggar_seni
        $data = [
            'id_sanggar' => $id_sanggar,
            'id_seni'    => $id_seni_baru
        ];

        // Memanggil fungsi baru di model
        $this->Pendaftaran_model->tambah_seni_ke_sanggar($data);

        $this->session->set_flashdata('success', 'Seni berhasil ditambahkan!');
        redirect('pendaftaran/biodata');
        return;
    }

    $data['sanggar']    = $sanggar;
    $data['jenis_seni'] = $this->Pendaftaran_model->get_all_jenis_seni();
    $this->load->view('pendaftaran/tambah_seni_sanggar', $data);
}
public function update_jenis_seni_single($id_sanggar, $id_seni_lama) {
    // Validasi kepemilikan sanggar seperti biasa
    // ...

    $this->form_validation->set_rules('jenis_seni_baru', 'Jenis Seni Baru', 'required');
    if ($this->form_validation->run() == FALSE) {
        $this->session->set_flashdata('error', validation_errors());
        $this->edit_jenis_seni_single($id_sanggar, $id_seni_lama);
        return;
    }

    $id_seni_baru = $this->input->post('jenis_seni_baru');

    // Panggil model untuk memperbarui
    $update = $this->Pendaftaran_model->update_jenis_seni_single($id_sanggar, $id_seni_lama, $id_seni_baru);

    if ($update) {
        $this->session->set_flashdata('success', 'Jenis seni berhasil diperbarui.');
    } else {
        $this->session->set_flashdata('error', 'Gagal memperbarui jenis seni.');
    }

    redirect('pendaftaran/biodata');
}
public function simpan_pendidikan()
{
    $data = [
        'id_sdm'        => $this->input->post('id_sdm'),
        'jenjang'       => $this->input->post('jenjang'),
        'program_studi' => $this->input->post('program_studi'),
        'nama_sekolah'  => $this->input->post('nama_sekolah'),
        'tahun_lulus'   => $this->input->post('tahun_lulus'),
    ];
    $this->db->insert('sdm_pendidikan', $data);

    echo json_encode(['status' => 'success']);
}

public function simpan_pelatihan()
{
    $id_sdm = $this->input->post('id_sdm');

    // Upload file sertifikat
    $bukti_sertifikat = $this->_uploadFile('bukti_sertifikat', './assets/uploads/pelatihan/');

    $data = [
        'id_sdm'           => $id_sdm,
        'nama_pelatihan'   => $this->input->post('nama_pelatihan'),
        'tingkat_pelatihan'=> $this->input->post('tingkat_pelatihan'),
        'tahun'            => $this->input->post('tahun'),
        'jenis_pelatihan'  => $this->input->post('jenis_pelatihan'),
        'peran'            => $this->input->post('peran'),
        'penyelenggara'    => $this->input->post('penyelenggara'),
        'bukti_sertifikat' => $bukti_sertifikat
    ];

    $this->db->insert('sdm_pelatihan', $data);

    echo json_encode(['status' => 'success']);
}


public function simpan_pekerjaan()
{
    $id_sdm = $this->input->post('id_sdm');

    $data = [
        'id_sdm'          => $id_sdm,
        'nama_pekerjaan'  => $this->input->post('nama_pekerjaan'), // FIX
        'nama_tempat_kerja'=> $this->input->post('nama_tempat_kerja'),
        'jabatan'         => $this->input->post('jabatan'),
        'status'          => $this->input->post('status'),
        'mulai_kerja'     => $this->input->post('mulai_kerja'),
        'selesai_kerja'   => $this->input->post('selesai_kerja')
    ];

    $this->db->insert('sdm_pekerjaan', $data);

    echo json_encode(['status' => 'success']);
}



public function simpan_aktivitas()
{
    $data = [
        'id_sdm'        => $this->input->post('id_sdm'),
        'nama_kegiatan' => $this->input->post('nama_kegiatan'),
        'peran'         => $this->input->post('peran'),
        'penyelenggara' => $this->input->post('penyelenggara'),
        'tingkat'       => $this->input->post('tingkat'),
        'tahun'         => $this->input->post('tahun'),
    ];
    $this->db->insert('sdm_aktivitas', $data);

    echo json_encode(['status' => 'success']);
}

public function simpan_penghargaan()
{
    $id_sdm = $this->input->post('id_sdm');

    // Upload file penghargaan
    $bukti_penghargaan = $this->_uploadFile('bukti_penghargaan', './assets/uploads/penghargaan/');

    $data = [
        'id_sdm'          => $id_sdm,
        'nama_penghargaan'=> $this->input->post('nama_penghargaan'),
        'tahun'           => $this->input->post('tahun'),
        'peran'           => $this->input->post('peran'),
        'pemberi'         => $this->input->post('pemberi'),
        'tingkat'         => $this->input->post('tingkat'),
        'bukti_penghargaan' => $bukti_penghargaan
    ];

    $this->db->insert('sdm_penghargaan', $data);

    echo json_encode(['status' => 'success']);
}

    private function _uploadFile($field_name, $path, $allowed_types = 'jpg|jpeg|png|pdf')
{
    $config['upload_path']   = $path;
    $config['allowed_types'] = $allowed_types;
    $config['max_size']      = 4096; // 4MB
    $config['encrypt_name']  = TRUE; // biar unik

    $this->load->library('upload', $config);

    if (!$this->upload->do_upload($field_name)) {
        return null; // gagal, return null
    } else {
        $data = $this->upload->data();
        return $data['file_name'];
    }
}
}